#include "t1001_rtc.h"

/**
 * @brief      Return the current system time in lp timer unit.
  *            This is a counter that grows since the power up of the system and it never wraps.
 * @return     Current system time.
 */
uint32_t RTC_GetCurrentSysTime(void)
{
    return SYS_CTRL->AON_RTC;
}

/**
 * @brief      Set the RTC Target value.
 * @param[in]  RTCx: where x can be 0 or 1 to select the RTC peripheral.
 * @param[in]  timeout: target value.
 */
void RTC_SetTimerTarget(uint32_t timeout)
{
    /** Check parameters valid. */

    SYS_CTRL->AON_RTC_TO = timeout;
}

/**
 * @brief      Get the RTC Target value.
 * @return     Target value.
 */
uint32_t RTC_GetTimerTarget(void)
{
    /** Check parameters valid. */

    return SYS_CTRL->AON_RTC_TO;
}

/**
 * @brief      Enables or disables the specified RTC interrupts.
 * @param[in]  RTCx: where x can be 0 or 1 to select the RTC peripheral.
 * @param[in]  NewState: new state of the specified RTCx interrupts.
 *               @arg ENABLE
 *               @arg DISABLE
 */
void RTC_ITConfig(uint16_t RTC_IT, FunctionalState NewState)
{
    /* Check the parameters */
    assert_param(IS_RTC_IT(RTC_IT));
    assert_param(IS_FUNCTIONAL_STATE(NewState));

    if (NewState != DISABLE)
    {
        /* Enable the selected RTC interrupt */
        SYS_CTRL->AON_RTC_IRQ |= RTC_IT;
    }
    else
    {
        /* Disable the selected RTC interrupt */
        SYS_CTRL->AON_RTC_IRQ &= ~RTC_IT;
    }
}

/**
 * @brief       Checks whether the RTC interrupt has occurred or not.
 * @param[in]   RTCx: where x can be 0 or 1 to select the RTC peripheral.
 * @return  The interrupt status.
 *              @arg SET
 *              @arg RESET
 */
ITStatus RTC_GetITStatus(uint16_t RTC_IT)
{
    /** Check parameter valid. */
    assert_param(IS_RTC_IT_STATUS(RTC_IT));

    ITStatus bitstatus = RESET;

    bitstatus = ((SYS_CTRL->AON_RTC_IRQ) & (RTC_IT == 0))? RESET: SET;
    return bitstatus;
}


/**
 * @brief       Clears the RTC's interrupt pending bits.
 * @param[in]   TIMx: where x can be 0 or 3 to select the RTC peripheral.
 * @param[in]   TIM_IT: Specifies the interrupt pending bit to clear.
 *              This parameter can be one of the following values:
 *                  @arg TIM_IT_Update:  RTC update Interrupt source
 */
void RTC_ClearITPendingBit(uint16_t RTC_IT)
{
    /** Check parameter valid. */
    assert_param(IS_RTC_IT_STATUS(RTC_IT));

    SYS_CTRL->AON_RTC_IRQ |= RTC_IT;
}
