/* ----------------------------------------------------------------------------
 * Copyright (c) 2020-2030 Boling Limited. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *   1. Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright notice,
 *      this list of conditions and the following disclaimer in the documentation
 *      and/or other materials provided with the distribution.
 *   3. Neither the name of Boling nor the names of its contributors
 *      may be used to endorse or promote products derived from this software
 *      without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * -------------------------------------------------------------------------- */

/**
 * @file     main.c
 * @brief    BL1826 CORE1 main entry
 * @date     10. Sept. 2021
 * @author   Boling SW Team
 *
 * @version
 * Version 1.0
 *  - Initial release
 *
 * @{
 */

/*******************************************************************************
 * INCLUDES
 */
#include "om.h"
#include "om_driver.h"
#include "evt.h"
#include "pm.h"
#include "board.h"
#include "mbr.h"
#include "nvds.h"

#include "ob_config.h"
#include "omble.h"
#include "app_common.h"

/*********************************************************************
 * MACROS
 */


/*********************************************************************
 * LOCAL VARIABLES
 */


/*********************************************************************
 * EXTERN FUNCTIONS
 */
void app_sec_init();
void service_common_init(void);
void app_blmicro_dfu_init(void);
void ancs_client_init(void);
void app_media_hid_init(void);
void app_tspp_init(void);
void app_shell_init(void);
extern void ob_smp_config_allroles(void);

/*********************************************************************
 * LOCAL FUNCTIONS
 */
/**
 *******************************************************************************
 * @brief  drv gpio isr handler
 *
 * @param[in] om_reg  om reg
 * @param[in] event  event
 * @param[in] param0  param0
 * @param[in] param1  param1
 *******************************************************************************
 */
static void drv_gpio_isr_handler(void *om_reg, drv_event_t event, void *int_status, void *data)
{
    OM_LOG_DEBUG("gpio: 0x%08X\n", (uint32_t)int_status);
}

/**
 *******************************************************************************
 * @brief  drv pin wakeup isr handler
 *
 * @param[in] om_reg  om reg
 * @param[in] event  event
 * @param[in] int_status  int status
 * @param[in] data  data
 *******************************************************************************
 */
static void drv_pin_wakeup_isr_handler(void *om_reg, drv_event_t event, void *int_status, void *data)
{
    OM_LOG_DEBUG("pinwakeup: 0x%08X\n", (uint32_t)int_status);
}

/**
 *******************************************************************************
 * @brief  pm sleep notify handler
 *
 * @param[in] sleep_state  sleep state
 * @param[in] power_status  power status
 *******************************************************************************
 */
static void pm_sleep_notify_handler(pm_sleep_state_t sleep_state, pm_status_t power_status)
{
    switch (sleep_state) {
        case PM_SLEEP_ENTRY:
            break;

        case PM_SLEEP_LEAVE_TOP_HALF:
            break;

        case PM_SLEEP_LEAVE_BOTTOM_HALF:
            break;

        default:
            break;
    }
}

/**
 *******************************************************************************
 * @brief  common init
 *******************************************************************************
 */
static void hardware_init(void)
{
    drv_gpio_register_isr_callback(OM_GPIO0, drv_gpio_isr_handler);

    drv_pmu_wakeup_pin_set(BITMASK(PAD_BUTTON_0) | BITMASK(PAD_BUTTON_1), PMU_PIN_WAKEUP_LOW_LEVEL);
    drv_pmu_wakeup_pin_register_callback(drv_pin_wakeup_isr_handler);

    pm_sleep_notify_user_callback_register(pm_sleep_notify_handler);
}

/*********************************************************************
 * CONST VARIABLES
 */

/*******************************************************************************
 * PUBLIC FUNCTIONS
 */

int main(void)
{
    struct ob_stack_param param = {
        .max_connection = OB_LE_HOST_CONNECTION_NB,
        .max_ext_adv_set = OB_LE_HOST_ADV_SET_NUM,
        .max_att_mtu = OB_LE_HOST_ATT_MTU,
        .max_gatt_serv_num = OB_LE_HOST_MAX_GATT_SERV_NUM,
        .max_gatt_write_cache = OB_LE_HOST_ATT_WRITE_CACHE_SIZE,
        .smp_sc_support = OB_LE_HOST_SC_PAIRING,
    };

    drv_wdt_init(0);
    board_init();
    hardware_init();
    drv_rf_init();
    evt_init();
    evt_timer_init();
    nvds_init(0);
    ob_smp_config_allroles();
    omble_init(&param);
    service_common_init();
    app_adv_init();
    app_sec_init();
    app_tspp_init();
    app_shell_init();
    app_conn_init();
    app_scan_init();
    app_gatt_client_init();

    uint32_t address, length;
    uint16_t crc16;
    mbr_read_part(PART_TYPE_APP, &address, &length, &crc16);
    OM_LOG_DEBUG("running(%d): 0x%08X, len: %d, crc: 0x%04X\n", drv_pmu_reboot_reason(), address, length, crc16);

    __enable_irq();

    // Main loop
    while (1) {
        // do event schedule
        evt_schedule();

        OM_CRITICAL_BEGIN();

        // if no event, do power manage
        if (evt_get_all() == 0) {
            pm_power_manage();
        }

        OM_CRITICAL_END();
    }
}


/** @} */
