/* ----------------------------------------------------------------------------
 * Copyright (c) 2020-2030 BoLing Limited. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *   1. Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright notice,
 *      this list of conditions and the following disclaimer in the documentation
 *      and/or other materials provided with the distribution.
 *   3. Neither the name of BoLing nor the names of its contributors
 *      may be used to endorse or promote products derived from this software
 *      without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * -------------------------------------------------------------------------- */

/**
 * @file     main.c
 * @brief    BL1826 CORE1 main entry
 * @date     10. Sept. 2021
 * @author   BoLing SW Team
 *
 * @version
 * Version 1.0
 *  - Initial release
 *
 * @{
 */

/*******************************************************************************
 * INCLUDES
 */
#include "om.h"
#include "om_driver.h"
#include "shell.h"
#include "evt.h"
#include "nvds.h"
#include "pm.h"
#include "board.h"

// Controller header
#include "obc.h"

/*********************************************************************
 * MACROS
 */
#define HCI_UART    OM_USART1

#define EVT_TYPE_HCI_H4  ((evt_type_t)(EVT_TYPE_USR_FIRST+0))

/*********************************************************************
 * LOCAL VARIABLES
 */
static om_kfifo_t hci_h4_rx_fifo;
static uint8_t hci_h4_rx_pool[512];
static obc_llt_t obc_llt_0;

static const usart_config_t hci_usart_cfg = {
    .baudrate       = 115200, //115200, // 1000000
    .flow_control   = USART_FLOW_CONTROL_NONE,
    .data_bit       = USART_DATA_BIT_8,
    .stop_bit       = USART_STOP_BIT_1,
    .parity         = USART_PARITY_NONE,
};

/*********************************************************************
 * LOCAL FUNCTIONS
 */

static void hci_h4_transmit_handler(const uint8_t *pdata, uint32_t length)
{
    drv_usart_write(HCI_UART, pdata, length, DRV_MAX_DELAY);
}

static void hci_h4_uart_rx_handler(void *om_usart, drv_event_t event, void *rx_buf, void *rx_cnt)
{
    if (event & DRV_EVENT_COMMON_READ_COMPLETED) {
        om_kfifo_in(&hci_h4_rx_fifo, rx_buf, (uint32_t)rx_cnt);
        evt_set(EVT_TYPE_HCI_H4);
    }
}

static void hci_h4_uart_rx_evt_handler(void)
{
    uint8_t buffer[64];
    uint32_t length;

    evt_clear(EVT_TYPE_HCI_H4);

    while (1) {

        length = om_kfifo_out(&hci_h4_rx_fifo, buffer, sizeof(buffer));

        if (length == 0)
            break;

        obc_hci_h4_receive_handler(buffer, length);
    }
}

static void ble_controller_init(void)
{
    obc_init();

    om_kfifo_init(&hci_h4_rx_fifo, hci_h4_rx_pool, sizeof(hci_h4_rx_pool));

    evt_callback_set(EVT_TYPE_HCI_H4, hci_h4_uart_rx_evt_handler);

    obc_hci_h4_transmit_callback_set(hci_h4_transmit_handler);
}

/**
 *******************************************************************************
 * @brief  drv gpio isr handler
 *
 * @param[in] om_reg  om reg
 * @param[in] event  event
 * @param[in] param0  param0
 * @param[in] param1  param1
 *******************************************************************************
 */
static void drv_gpio_isr_handler(void *om_reg, drv_event_t event, void *int_status, void *data)
{
    OM_LOG_DEBUG("gpio: 0x%08X\n", (uint32_t)int_status);
}

/**
 *******************************************************************************
 * @brief  drv pin wakeup isr handler
 *
 * @param[in] om_reg  om reg
 * @param[in] event  event
 * @param[in] int_status  int status
 * @param[in] data  data
 *******************************************************************************
 */
static void drv_pin_wakeup_isr_handler(void *om_reg, drv_event_t event, void *int_status, void *data)
{
    OM_LOG_DEBUG("pinwakeup: 0x%08X\n", (uint32_t)int_status);
}

/**
 *******************************************************************************
 * @brief  pm sleep notify handler
 *
 * @param[in] sleep_state  sleep state
 * @param[in] power_status  power status
 *******************************************************************************
 */
static void pm_sleep_notify_handler(pm_sleep_state_t sleep_state, pm_status_t power_status)
{
    switch(sleep_state)
    {
        case PM_SLEEP_ENTRY:
            break;

        case PM_SLEEP_LEAVE_TOP_HALF:
            drv_usart_init(HCI_UART, &hci_usart_cfg);
            drv_usart_read_int(HCI_UART, NULL, 0);
            break;

        case PM_SLEEP_LEAVE_BOTTOM_HALF:
            break;

        default:
            break;
    }
}

/**
 *******************************************************************************
 * @brief  common init
 *******************************************************************************
 */
static void hardware_init(void)
{
    drv_usart_init(HCI_UART, &hci_usart_cfg);
    drv_usart_register_isr_callback(HCI_UART, hci_h4_uart_rx_handler);
    drv_usart_read_int(HCI_UART, NULL, 0);

    drv_gpio_register_isr_callback(OM_GPIO0, drv_gpio_isr_handler);

    drv_pmu_wakeup_pin_set(BITMASK(PAD_BUTTON_0)|BITMASK(PAD_BUTTON_1), PMU_PIN_WAKEUP_LOW_LEVEL);
    drv_pmu_wakeup_pin_register_callback(drv_pin_wakeup_isr_handler);

    pm_sleep_notify_user_callback_register(pm_sleep_notify_handler);
}

static obc_llt_status_t obc_llt_0_event_handler(obc_llt_event_t event, uint32_t cur_time, uint32_t *next_trigger_time)
{
//    OM_LOG_DEBUG("LLT: %d\n", event);

    return OBC_LLT_STATUS_CONTINUE;
}

void cmd_shell_llt(int argc, char *argv[])
{
    if (strcmp(argv[0], "start") == 0) {
        if (argc < 4) {
            OM_LOG_DEBUG("LLT: invalid params\n");
            return;
        }

        obc_llt_params_t params;
        params.mode         = strcmp(argv[1], "ONTIME")==0  ? OBC_LLT_MODE_ONTIME :
                              strcmp(argv[1], "NONTIME")==0 ? OBC_LLT_MODE_NOT_ONTIME : OBC_LLT_MODE_BACKGROUND;
        params.prio_init    = strtoul(argv[2], NULL, 0);
        params.prio_inc     = 8;
        params.interval_min = strtoul(argv[3], NULL, 0);
        params.interval_max = params.interval_min;
        params.duration_min = strtoul(argv[4], NULL, 0);
        params.duration_max = params.duration_min;
        params.event_cb     = obc_llt_0_event_handler;

        OM_LOG_DEBUG("LLT: mode=%d, prio=%d, intv=0x%x duration=0x%x\n", params.mode, params.prio_init, params.interval_min, params.duration_min);

        obc_llt_init(&obc_llt_0);
        obc_llt_start(&obc_llt_0);
        obc_llt_params_set(&obc_llt_0, &params);
        obc_llt_timer_set(&obc_llt_0, OBC_LLT_AUTO_TIME);

    } else if (strcmp(argv[0], "stop") == 0) {
        obc_llt_stop(&obc_llt_0);

    } else {
        OM_LOG_DEBUG("LLT: invalid command\n");
    }
}

void cmd_shell_pta(int argc, char *argv[])
{
    if (strcmp(argv[0], "start") == 0) {
        if (argc < 1) {
            OM_LOG_DEBUG("LLT: invalid params\n");
            return;
        }

        DRV_PIN_MUX_SET(2, PINMUX_PTA_ACTIVE_IN_CFG);
        DRV_PIN_MUX_SET(15, PINMUX_PTA_ACTIVE_OUT_CFG);
        DRV_PIN_MUX_SET(16, PINMUX_PTA_PRIORITY_CFG);
        DRV_PIN_MUX_SET(17, PINMUX_PTA_FREQ_CFG);

        obc_pta_ctrl_t ctrl;
        ctrl.grant_active_level = 0;
        ctrl.priority_txrx_include = 1;
        ctrl.priority_txrx_tx_active_level = 1;
        ctrl.priority_prio_active_keep_us = 20;
        ctrl.priority_txrx_active_keep_us = 30;
        ctrl.priority_threshold = strtoul(argv[1], NULL, 0);
        ctrl.priority_state_initiating_connection_ind_rsp = 2;
        ctrl.priority_state_connection_llcp = 15;
        ctrl.priority_state_connection_data_channel = 10;
        ctrl.priority_state_initiating_scanning = 5;
        ctrl.priority_state_active_scanning = 2;
        ctrl.priority_state_connectable_advertising = 4;
        ctrl.priority_state_non_connectable_advertising = 4;
        ctrl.priority_state_passive_scanning = 2;

        OM_LOG_DEBUG("PTA: priority_threshold=%d\n", ctrl.priority_threshold);

        obc_pta_enable(true, &ctrl);

    } else if (strcmp(argv[0], "stop") == 0) {
        obc_pta_enable(false, NULL);

    } else {
        OM_LOG_DEBUG("PTA: invalid command\n");
    }
}

/*********************************************************************
 * CONST VARIABLES
 */

const static shell_cmd_t ble_hci_shell_cmd[] = {
    { "llt",     cmd_shell_llt,     "llt start/stop <mode(ONTIME/NONTIME/BG)> <prio> <intv> <duration>" },
    { "pta",     cmd_shell_pta,     "pta <priority_threshold>" },
    { NULL,      NULL,              NULL},     /* donot deleted */
};

/*******************************************************************************
 * PUBLIC FUNCTIONS
 */
int main(void)
{
    drv_wdt_init(0);
    board_init();
    hardware_init();
    drv_rf_init();
    evt_init();
    evt_timer_init();
    shell_init(ble_hci_shell_cmd);
    nvds_init(0);
    ble_controller_init();

    __enable_irq();

    // Main loop
    while(1) {
        // do event schedule
        evt_schedule();

        OM_CRITICAL_BEGIN();

        // if no event, do power manage
        if (evt_get_all() == 0) {
            pm_power_manage();
        }

        OM_CRITICAL_END();
    }
}


/** @} */
