/* ----------------------------------------------------------------------------
 * Copyright (c) 2020-2030 Boling Limited. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *   1. Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright notice,
 *      this list of conditions and the following disclaimer in the documentation
 *      and/or other materials provided with the distribution.
 *   3. Neither the name of Boling nor the names of its contributors
 *      may be used to endorse or promote products derived from this software
 *      without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * -------------------------------------------------------------------------- */

/**
 * @file     app_om_cgms.c
 * @date     30. Dec. 2023
 * @author   Boling SW Team
 *
 * @version
 * Version 1.0
 *  - Initial release
 *
 * @{
 */

/*******************************************************************************
 * INCLUDES
 */
#include <stdint.h>
#include "omble.h"
#include "om_cgms.h"
#include "service_om_cgms.h"

/*********************************************************************
 * LOCAL VARIABLES
 */
static int cgms_record_num;
static uint8_t cgms_interval;

static void get_session_start_time(om_cgm_time_t *time, int8_t *time_zone, uint8_t *dst_offset, void *p);
static void set_session_start_time(const om_cgm_time_t *time, int8_t time_zone, uint8_t dst_offset, void *p);
static uint16_t read_session_run_time(void *p);
static void get_status(om_cgm_state_t *status, uint16_t *time_offset, void *p);
static bool set_comm_interval(uint8_t interval, void *p);
static void get_comm_interval(uint8_t *interval, void *p);
static uint16_t get_records_number(uint8_t type, uint16_t time_offset, void *p);
static bool get_records(int index, uint8_t type, uint16_t time_offset,
                        om_cgms_measurement_t *data, om_cgm_state_t *state, void *p);

const static om_cgms_handlers_t cgms_cbs = {
    get_session_start_time,
    set_session_start_time,
    read_session_run_time,
    get_status,
    set_comm_interval,
    get_comm_interval,
    get_records_number,
    get_records,
};

/*********************************************************************
 * LOCAL FUNCTIONS
 */
static void get_session_start_time(om_cgm_time_t *time, int8_t *time_zone, uint8_t *dst_offset, void *p)
{
    time->year = 2023, time->month = 12, time->day = 22;
    time->hours = 17, time->minutes = 30, time->seconds = 59;
    *time_zone = -40;
    *dst_offset = 0;
}

static void set_session_start_time(const om_cgm_time_t *time, int8_t time_zone, uint8_t dst_offset, void *p)
{
}

static uint16_t read_session_run_time(void *p)
{
    return 100;
}

static void get_status(om_cgm_state_t *status, uint16_t *time_offset, void *p)
{
}

static bool set_comm_interval(uint8_t interval, void *p)
{
    cgms_interval = interval;
    return true;
}

static void get_comm_interval(uint8_t *interval, void *p)
{
    *interval = cgms_interval;
}

static uint16_t get_records_number(uint8_t type, uint16_t time_offset, void *p)
{
    if (type == CGM_RACP_VALUE_ALL_RECORDS) {
        return cgms_record_num;
    } else { // CGM_RACP_VALUE_GREATER_EQUAL
        if (time_offset > cgms_record_num * 10) {
            return 0;
        }
        return (cgms_record_num * 10 - time_offset) / 10;
    }
}

static bool get_records(int index, uint8_t type, uint16_t time_offset,
                        om_cgms_measurement_t *data, om_cgm_state_t *state, void *p)
{
    if (type == CGM_RACP_VALUE_LAST_RECORD) {
        data->glucose_concentration = 1.234;
        data->time_offset = (cgms_record_num - 1) * 10;
        return true;
    }
    if (time_offset > (cgms_record_num - 1 - index) * 10) {
        return false;
    }
    state->device_battery_low = 1;
    data->glucose_concentration = +1.2340;
    data->time_offset = time_offset + index * 10;
    return index < cgms_record_num ? true : false;
}

/*******************************************************************************
 * PUBLIC FUNCTIONS
 */
void app_om_cgms_init(void)
{
    service_om_cgms_init();
    om_cgms_config_t config = {
        CGM_LOCATION_FINGER,
        CGM_TYPE_CAPILLARY_WHOLE_BLOOD,
    };
    om_cgms_init(&config, &cgms_cbs, NULL);
    cgms_record_num = 5;
}

/** @} */
