/* ----------------------------------------------------------------------------
 * Copyright (c) 2020-2030 BoLing Limited. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *   1. Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright notice,
 *      this list of conditions and the following disclaimer in the documentation
 *      and/or other materials provided with the distribution.
 *   3. Neither the name of BoLing nor the names of its contributors
 *      may be used to endorse or promote products derived from this software
 *      without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * -------------------------------------------------------------------------- */

/**
 * @file     om24g.c
 * @brief    OM24G driver
 * @date     1. Dec. 2022
 * @author   BoLing SW Team
 *
 * @version
 * Version 1.0
 *  - Initial release
 *
 * @{
 */


/*******************************************************************************
 * INCLUDES
 */
#include "RTE_bl182x.h"
#if (RTE_OM24G)
#include <stddef.h>
#include "om_device.h"
#include "om_driver.h"
/*********************************************************************
 * MACROS
 */
#define PLL_CONTIMUE_OPEN 0 
/*******************************************************************************
 * TYPEDEFS
 */
typedef struct {
    drv_isr_callback_t           event_cb;          /**< event callback                      */
    uint16_t                     tx_cnt;           /**< Count of data sent*/
    uint16_t                     max_rx_num;        /**< num of data sent*/
    const uint8_t                *tx_buf;            /**< Pointer to out data buffer, system to peripheral direction */
    uint8_t                      *rx_buf;            /**< Pointer to in data buffer            */
    bool                         deviation_below_250K;/**< Flag with frequency deviation less than 250K */
    struct {
        uint32_t RX_GFSK_SYNC_CTRL;
        uint32_t TX_CTRL0;
        uint32_t PLL_CTRL1;
        uint32_t FREQ_CFG3;
        uint32_t FREQ_CFG0;
        uint32_t PKTCTRL0;
        uint32_t FB_PKTCTRL;
        uint32_t SYNC_WORD0;
        uint32_t SYNC_WORD1;
        uint32_t DMA_TX_ADDR;
        uint32_t DMA_RX_ADDR;
        uint32_t CRCPOLY;
        uint32_t CRCINIT;
        uint32_t SETUP_VALUE;
        uint32_t RXCRC;
        uint32_t PACKET_LEN;
        uint8_t DYNPD;
        uint8_t FEATURE;
        uint8_t INT_MASK;
        uint8_t EN_AA;
        uint8_t EN_RXADDR;
        uint8_t FA_PKTCTRL;
        uint8_t FA_SETUP_RETR;
        uint8_t PREAMBLE;
        uint8_t PREAMBLE_LEN;
        uint16_t PRE_GUARD;
        uint8_t TX_ADDR;
        uint8_t SYNC_WORD_SEL;
        uint8_t RX_ADDR_P0;
        uint8_t RX_ADDR_P1;
        uint8_t RX_ADDR_P2;
        uint8_t RX_ADDR_P3;
        uint8_t RX_ADDR_P4;
        uint8_t RX_ADDR_P5;
        uint8_t RX_ADDR_P6;
        uint8_t RX_ADDR_P7;
        uint16_t RF_DR;
        uint8_t CRCCFG;
        uint8_t CRCSKIP;
        uint8_t WHITECFG;
        uint8_t WHITESEL;
        uint8_t WHITEOUT;
        uint8_t ACK_MODE;
        uint8_t FPGA_TX_INDEX;
        uint16_t WHITESEED;
        uint16_t ENDIAN;
        uint8_t FD_CFO_CMP;
        uint8_t DET_MODE;
        uint16_t H_RX_CTRL;
        uint16_t RF_PD_AHEAD;
        uint32_t KDCO_LUT_2M_REG0;
        uint32_t KDCO_LUT_2M_REG1;
        uint32_t PD_CFG2;
        uint32_t STR_CTRL;
    } reg_store;
} om24g_env_t;



/*******************************************************************************
 * CONST & VARIABLES
 */

/* om24g information */
#if (RTE_OM24G)
static om24g_env_t om24g_env = {
    .event_cb      = NULL,
    .tx_cnt        = 0U,
    .tx_buf        = NULL,
    .rx_buf        = NULL,
    .deviation_below_250K = false,
};

static uint8_t devia_offset[22][2] = { {0xA, 0x1}, {0xB, 0x1}, {0xC, 0x2}, {0xD, 0x2}, {0xE, 0x3}, {0xF, 0x3}, {0x10, 0x4}, {0x11, 0x4},
                                      {0x12, 0x5}, {0x13, 0x5}, {0x14, 0x6}, {0x15, 0x6}, {0x16, 0x7}, {0x17, 0x7}, {0x18, 0x8}, {0x19, 0x8}, 
                                      {0x1A, 0x9}, {0x1B, 0x9}, {0x1C, 0xA}, {0x1D, 0xA}, {0x1E, 0xB}, {0x1E, 0xB}};
#endif  /* RTE_OM24G */

/*******************************************************************************
 * LOCAL FUNCTIONS
 */
__RAM_CODES("24G")
static void om24g_ce_high_pulse(void)
{
    OM24G_CE_HIGH();
    DRV_DELAY_US(1);
    OM_24G->DMA_CMD = 0xA0;
    DRV_DELAY_US(3);
    OM24G_CE_LOW();
}

__RAM_CODES("24G")
static uint16_t om24g_read_rx_payload_width(void)
{
    uint16_t Width;
    uint8_t en_dpl;
    en_dpl = REGR(&OM_24G->FEATURE, OM24G_EN_DPL_MASK, OM24G_EN_DPL_POS);
    if (en_dpl) {
        Width = REGR(&OM_24G->RX_DYN_LEN, OM24G_RX_DYN_LEN_MASK, OM24G_RX_DYN_LEN_POS);
    } else {
        Width = REGR(&OM_24G->PACKET_LEN, OM24G_PACKET_LEN_MASK, OM24G_PACKET_LEN_POS);
    }
    return Width;
}

__RAM_CODES("24G")
static void om24g_reg_store(void)
{
    om24g_control(OM24G_CONTROL_CLK_ENABLE, NULL);
    om24g_env.reg_store.PKTCTRL0 = OM_24G->PKTCTRL0;
    om24g_env.reg_store.FB_PKTCTRL = OM_24G->FB_PKTCTRL;
    om24g_env.reg_store.DYNPD = OM_24G->DYNPD;
    om24g_env.reg_store.FEATURE = OM_24G->FEATURE;
    om24g_env.reg_store.INT_MASK = OM_24G->INT_MASK;
    om24g_env.reg_store.EN_AA = OM_24G->EN_AA;
    om24g_env.reg_store.EN_RXADDR = OM_24G->EN_RXADDR;
    om24g_env.reg_store.FA_PKTCTRL = OM_24G->FA_PKTCTRL;
    om24g_env.reg_store.FA_SETUP_RETR = OM_24G->FA_SETUP_RETR;
    om24g_env.reg_store.PREAMBLE = OM_24G->PREAMBLE;
    om24g_env.reg_store.PREAMBLE_LEN = OM_24G->PREAMBLE_LEN;
    om24g_env.reg_store.PRE_GUARD = OM_24G->PRE_GUARD;
    om24g_env.reg_store.SYNC_WORD0 = OM_24G->SYNC_WORD0;
    om24g_env.reg_store.SYNC_WORD1 = OM_24G->SYNC_WORD1;
    om24g_env.reg_store.TX_ADDR = OM_24G->TX_ADDR;
    om24g_env.reg_store.SYNC_WORD_SEL = OM_24G->SYNC_WORD_SEL;
    om24g_env.reg_store.RX_ADDR_P0 = OM_24G->RX_ADDR_P0;
    om24g_env.reg_store.RX_ADDR_P1 = OM_24G->RX_ADDR_P1;
    om24g_env.reg_store.RX_ADDR_P2 = OM_24G->RX_ADDR_P2;
    om24g_env.reg_store.RX_ADDR_P3 = OM_24G->RX_ADDR_P3;
    om24g_env.reg_store.RX_ADDR_P4 = OM_24G->RX_ADDR_P4;
    om24g_env.reg_store.RX_ADDR_P5 = OM_24G->RX_ADDR_P5;
    om24g_env.reg_store.RX_ADDR_P6 = OM_24G->RX_ADDR_P6;
    om24g_env.reg_store.RX_ADDR_P7 = OM_24G->RX_ADDR_P7;
    om24g_env.reg_store.RF_DR = OM_24G->RF_DR;              
    om24g_env.reg_store.CRCCFG = OM_24G->CRCCFG;
    om24g_env.reg_store.CRCPOLY = OM_24G->CRCPOLY;
    om24g_env.reg_store.CRCINIT = OM_24G->CRCINIT;
    om24g_env.reg_store.CRCSKIP = OM_24G->CRCSKIP;
    om24g_env.reg_store.WHITECFG = OM_24G->WHITECFG;
    om24g_env.reg_store.WHITESEL = OM_24G->WHITESEL;
    om24g_env.reg_store.WHITESEED = OM_24G->WHITESEED;
    om24g_env.reg_store.WHITEOUT = OM_24G->WHITEOUT;
    om24g_env.reg_store.DMA_TX_ADDR = OM_24G->DMA_TX_ADDR;
    om24g_env.reg_store.DMA_RX_ADDR = OM_24G->DMA_RX_ADDR;
    om24g_env.reg_store.PACKET_LEN = OM_24G->PACKET_LEN;
    om24g_env.reg_store.SETUP_VALUE = OM_24G->SETUP_VALUE;
    om24g_env.reg_store.ENDIAN = OM_24G->ENDIAN;
    om24g_env.reg_store.ACK_MODE = OM_24G->ACK_MODE;
    om24g_env.reg_store.FPGA_TX_INDEX = OM_24G->FPGA_TX_INDEX;
    om24g_env.reg_store.FREQ_CFG0 = OM_DAIF->FREQ_CFG0;
    om24g_env.reg_store.FREQ_CFG3 = OM_DAIF->FREQ_CFG3;
    om24g_env.reg_store.PLL_CTRL1 = OM_DAIF->PLL_CTRL1;
    om24g_env.reg_store.PD_CFG2 = OM_DAIF->PD_CFG2;
    om24g_env.reg_store.H_RX_CTRL = OM_PHY->H_RX_CTRL;
    om24g_env.reg_store.FD_CFO_CMP = OM_PHY->FD_CFO_CMP;
    om24g_env.reg_store.DET_MODE = OM_PHY->DET_MODE;
    om24g_env.reg_store.TX_CTRL0 = OM_PHY->TX_CTRL0;
    om24g_env.reg_store.RX_GFSK_SYNC_CTRL = OM_PHY->RX_GFSK_SYNC_CTRL;
    om24g_env.reg_store.STR_CTRL = OM_PHY->STR_CTRL;
    om24g_env.reg_store.RF_PD_AHEAD = OM_24G->RF_PD_AHEAD;
    om24g_control(OM24G_CONTROL_CLK_DISABLE, NULL);
}

__RAM_CODES("24G")
static void om24g_reg_restore(void)
{
    om24g_control(OM24G_CONTROL_CLK_ENABLE, NULL);
    OM_24G->PKTCTRL0 = om24g_env.reg_store.PKTCTRL0;
    OM_24G->PRE_GUARD = om24g_env.reg_store.PRE_GUARD;
    OM_24G->PREAMBLE = om24g_env.reg_store.PREAMBLE;
    OM_24G->PREAMBLE_LEN = om24g_env.reg_store.PREAMBLE_LEN;
    OM_24G->SYNC_WORD_SEL = om24g_env.reg_store.SYNC_WORD_SEL;
    OM_24G->SYNC_WORD0 = om24g_env.reg_store.SYNC_WORD0;
    OM_24G->SYNC_WORD1 = om24g_env.reg_store.SYNC_WORD1;
    OM_24G->TX_ADDR = om24g_env.reg_store.TX_ADDR;
    OM_24G->RX_ADDR_P0 = om24g_env.reg_store.RX_ADDR_P0;
    OM_24G->RX_ADDR_P1 = om24g_env.reg_store.RX_ADDR_P1;
    OM_24G->RX_ADDR_P2 = om24g_env.reg_store.RX_ADDR_P2;
    OM_24G->RX_ADDR_P3 = om24g_env.reg_store.RX_ADDR_P3;
    OM_24G->RX_ADDR_P4 = om24g_env.reg_store.RX_ADDR_P4;
    OM_24G->RX_ADDR_P5 = om24g_env.reg_store.RX_ADDR_P5;
    OM_24G->RX_ADDR_P6 = om24g_env.reg_store.RX_ADDR_P6;
    OM_24G->RX_ADDR_P7 = om24g_env.reg_store.RX_ADDR_P7;
    OM_24G->PACKET_LEN = om24g_env.reg_store.PACKET_LEN;    
    OM_24G->FB_PKTCTRL = om24g_env.reg_store.FB_PKTCTRL;
    OM_24G->RF_DR = om24g_env.reg_store.RF_DR;
    OM_DAIF->FREQ_CFG0 = om24g_env.reg_store.FREQ_CFG0;
    OM_DAIF->FREQ_CFG3 = om24g_env.reg_store.FREQ_CFG3;
    OM_DAIF->PLL_CTRL1 = om24g_env.reg_store.PLL_CTRL1;
    OM_DAIF->PD_CFG2 = om24g_env.reg_store.PD_CFG2;
    OM_24G->FPGA_TX_INDEX = om24g_env.reg_store.FPGA_TX_INDEX;
    OM_PHY->H_RX_CTRL = om24g_env.reg_store.H_RX_CTRL;
    OM_24G->EN_AA = om24g_env.reg_store.EN_AA;
    OM_24G->FEATURE = om24g_env.reg_store.FEATURE;
    OM_24G->DYNPD = om24g_env.reg_store.DYNPD;
    OM_24G->WHITECFG = om24g_env.reg_store.WHITECFG;
    OM_24G->WHITESEL = om24g_env.reg_store.WHITESEL;
    OM_24G->WHITESEED = om24g_env.reg_store.WHITESEED;
    OM_24G->WHITEOUT = om24g_env.reg_store.WHITEOUT;
    OM_24G->CRCCFG = om24g_env.reg_store.CRCCFG;
    OM_24G->CRCPOLY = om24g_env.reg_store.CRCPOLY;
    OM_24G->CRCINIT = om24g_env.reg_store.CRCINIT;
    OM_24G->CRCSKIP = om24g_env.reg_store.CRCSKIP;
    OM_24G->ENDIAN = om24g_env.reg_store.ENDIAN;
    OM_24G->FA_SETUP_RETR = om24g_env.reg_store.FA_SETUP_RETR;
    OM_24G->DMA_TX_ADDR = om24g_env.reg_store.DMA_TX_ADDR;
    OM_24G->DMA_RX_ADDR = om24g_env.reg_store.DMA_RX_ADDR;
    OM_24G->INT_MASK = om24g_env.reg_store.INT_MASK;
    OM_24G->SETUP_VALUE = om24g_env.reg_store.SETUP_VALUE;
    OM_PHY->TX_CTRL0 = om24g_env.reg_store.TX_CTRL0;
    OM_PHY->FD_CFO_CMP = om24g_env.reg_store.FD_CFO_CMP;
    OM_PHY->DET_MODE = om24g_env.reg_store.DET_MODE;
    OM_PHY->RX_GFSK_SYNC_CTRL = om24g_env.reg_store.RX_GFSK_SYNC_CTRL;
    OM_PHY->STR_CTRL = om24g_env.reg_store.STR_CTRL;
    OM_24G->FA_PKTCTRL = om24g_env.reg_store.FA_PKTCTRL;
    OM_24G->EN_RXADDR = om24g_env.reg_store.EN_RXADDR;
    OM_24G->ACK_MODE = om24g_env.reg_store.ACK_MODE;
    OM_24G->RF_PD_AHEAD = om24g_env.reg_store.RF_PD_AHEAD;
    om24g_control(OM24G_CONTROL_CLK_DISABLE, NULL);
}

#if (RTE_PM)
__RAM_CODES("24G")
static void om24g_pm_sleep_store_restore_handler(pm_sleep_state_t sleep_state, pm_status_t power_status)
{
    if(sleep_state == PM_SLEEP_STORE)
    {
        om24g_reg_store();
        //OM_LOG_DEBUG("om24g_reg_store\r\n");
    }
    else if (sleep_state == PM_SLEEP_RESTORE_HSI)
    {
        om24g_reg_restore();
        //OM_LOG_DEBUG("om24g_reg_restore_HSI\r\n");
    }
    else if (sleep_state == PM_SLEEP_RESTORE_HSE)
    {
        //om24g_reg_restore();
        //OM_LOG_DEBUG("om24g_reg_restore_HSE\r\n");
    }
}
#endif

static void om24g_dump_rf_register(void)
{
    OM_LOG_DEBUG("PKTCTRL0: %08x\r\n", OM_24G->PKTCTRL0);
    OM_LOG_DEBUG("FB_PKTCTRL: %08x\r\n", OM_24G->FB_PKTCTRL);
    OM_LOG_DEBUG("DYNPD: %02x\r\n", OM_24G->DYNPD);
    OM_LOG_DEBUG("FEATURE: %02x\r\n", OM_24G->FEATURE);
    OM_LOG_DEBUG("INT_ST: %02x\r\n", OM_24G->INT_ST);
    OM_LOG_DEBUG("EN_AA: %02x\r\n", OM_24G->EN_AA);
    OM_LOG_DEBUG("EN_RXADDR: %02x\r\n", OM_24G->EN_RXADDR);
    OM_LOG_DEBUG("FA_PKTCTRL: %02x\r\n", OM_24G->FA_PKTCTRL);
    OM_LOG_DEBUG("SETUP_RETR: %02x\r\n", OM_24G->FA_SETUP_RETR);
    OM_LOG_DEBUG("FA_OBSERVE_TX: %02x\r\n", OM_24G->FA_OBSERVE_TX);
    OM_LOG_DEBUG("PREAMBLE: %02x\r\n", OM_24G->PREAMBLE);
    OM_LOG_DEBUG("PREAMBLE_LEN: %02x\r\n", OM_24G->PREAMBLE_LEN);
    OM_LOG_DEBUG("PRE_GUARD: %02x\r\n", OM_24G->PRE_GUARD);
    OM_LOG_DEBUG("SYNC_WORD0: %08x\r\n", OM_24G->SYNC_WORD0);
    OM_LOG_DEBUG("SYNC_WORD1: %08x\r\n", OM_24G->SYNC_WORD1);
    OM_LOG_DEBUG("TX_ADDR: %02x\r\n", OM_24G->TX_ADDR);
    OM_LOG_DEBUG("SYNC_WORD_SEL: %02x\r\n", OM_24G->SYNC_WORD_SEL);
    OM_LOG_DEBUG("RX_ADDR_P0: %02x\r\n", OM_24G->RX_ADDR_P0);
    OM_LOG_DEBUG("RX_ADDR_P1: %02x\r\n", OM_24G->RX_ADDR_P1);
    OM_LOG_DEBUG("RX_ADDR_P2: %02x\r\n", OM_24G->RX_ADDR_P2);
    OM_LOG_DEBUG("RX_ADDR_P3: %02x\r\n", OM_24G->RX_ADDR_P3);
    OM_LOG_DEBUG("RX_ADDR_P4: %02x\r\n", OM_24G->RX_ADDR_P4);
    OM_LOG_DEBUG("RX_ADDR_P5: %02x\r\n", OM_24G->RX_ADDR_P5);
    OM_LOG_DEBUG("RX_ADDR_P6: %02x\r\n", OM_24G->RX_ADDR_P6);
    OM_LOG_DEBUG("RX_ADDR_P7: %02x\r\n", OM_24G->RX_ADDR_P7);
    OM_LOG_DEBUG("RF_DR: %02x\r\n", OM_24G->RF_DR);
    OM_LOG_DEBUG("RX_P_NO: %02x\r\n", OM_24G->RX_P_NO);
    OM_LOG_DEBUG("CRCCFG: %08x\r\n", OM_24G->CRCCFG);
    OM_LOG_DEBUG("CRCPOLY: %08x\r\n", OM_24G->CRCPOLY);
    OM_LOG_DEBUG("CRCINIT: %08x\r\n", OM_24G->CRCINIT);
    OM_LOG_DEBUG("CRCSKIP: %02x\r\n", OM_24G->CRCSKIP);
    OM_LOG_DEBUG("WHITECFG: %02x\r\n", OM_24G->WHITECFG);
    OM_LOG_DEBUG("WHITESEL: %02x\r\n", OM_24G->WHITESEL);
    OM_LOG_DEBUG("WHITESEED: %02x\r\n", OM_24G->WHITESEED);
    OM_LOG_DEBUG("WHITEOUT: %02x\r\n", OM_24G->WHITEOUT);
    OM_LOG_DEBUG("DMA_CMD: %02x\r\n", OM_24G->DMA_CMD);
    OM_LOG_DEBUG("DMA_TX_LEN: %02x\r\n", OM_24G->DMA_TX_LEN);
    OM_LOG_DEBUG("RX_DYN_LEN: %02x\r\n", OM_24G->RX_DYN_LEN);
    OM_LOG_DEBUG("DMA_TX_ADDR: %02x\r\n", OM_24G->DMA_TX_ADDR);
    OM_LOG_DEBUG("DMA_RX_ADDR: %08x\r\n", OM_24G->DMA_RX_ADDR);
    OM_LOG_DEBUG("PACKET_LEN: %02x\r\n", OM_24G->PACKET_LEN);
    OM_LOG_DEBUG("BCC: %02x\r\n", OM_24G->BCC);
    OM_LOG_DEBUG("TIMESTAMP_RT: %02x\r\n", OM_24G->TIMESTAMP_RT);
    OM_LOG_DEBUG("TIMESTAMP_TRIGER: %02x\r\n", OM_24G->TIMESTAMP_TRIGER);
    OM_LOG_DEBUG("TIMESTAMP_CFG: %02x\r\n", OM_24G->TIMESTAMP_CFG);
    OM_LOG_DEBUG("SETUP_VALUE: %02x\r\n", OM_24G->SETUP_VALUE);
    OM_LOG_DEBUG("ENDIAN: %02x\r\n", OM_24G->ENDIAN);
    OM_LOG_DEBUG("FLUSH: %02x\r\n", OM_24G->FLUSH);
    OM_LOG_DEBUG("TESTCTRL: %02x\r\n", OM_24G->TESTCTRL);
    OM_LOG_DEBUG("STATE: %02x\r\n", OM_24G->STATE);
    OM_LOG_DEBUG("FREQ: %02d\r\n", REGR(&OM_DAIF->FREQ_CFG0, DAIF_FREQ_REG_MO_MASK, DAIF_FREQ_REG_MO_POS));
    OM_LOG_DEBUG("FRACT_FREQ: %x\r\n", OM_DAIF->FREQ_CFG3);
    OM_LOG_DEBUG("OM_PHY->REG_PHY_RST_N: %02x\r\n", OM_PHY->REG_PHY_RST_N);
    OM_LOG_DEBUG("OM_PHY->TX_CTRL0: %02x\r\n", OM_PHY->TX_CTRL0);
    OM_LOG_DEBUG("OM_PHY->DET_MODE: %02x\r\n", OM_PHY->DET_MODE);
    OM_LOG_DEBUG("OM_PHY->H_RX_CTRL: %02x\r\n", OM_PHY->H_RX_CTRL);
    OM_LOG_DEBUG("OM_PHY->FD_CFO_CMP: %02x\r\n", OM_PHY->FD_CFO_CMP);
    OM_LOG_DEBUG("OM_PHY->STR_CTRL: %02x\r\n", OM_PHY->STR_CTRL);
    OM_LOG_DEBUG("OM_PHY->RX_GFSK_SYNC_CTRL: %08x\r\n", OM_PHY->RX_GFSK_SYNC_CTRL);

}

static uint8_t om24g_set_kdco(uint8_t kdco_value)
{
    if(kdco_value >= 0xA) {
        for (uint8_t i = 0; i < 22; i++) {
            if (kdco_value == devia_offset[i][0]) {
                return kdco_value = devia_offset[i][1];
            }
        }     
    }
    return kdco_value;
}

static void om24g_set_deviation(om24g_deviation_t deviation)
{
    if (REGR(&OM_24G->FEATURE, MASK_POS(OM24G_EN_ACK_PAY)) == 0x1) {
        deviation = OM24G_DEVIATION_250K;
    }
    switch (deviation) {
        case OM24G_DEVIATION_62P5K:
            om24g_env.deviation_below_250K = false;
            REGW0(&OM_DAIF->PLL_CTRL1, DAIF_FREQ_DEVIA_BYPASS_MASK); //freq devia bypass=0d
            REGW(&OM_DAIF->PLL_CTRL1, MASK_1REG(DAIF_FREQ_DEVIA_COEFF, 0xFF)); // TX deviation value = 62.5k
            REGW(&OM_DAIF->PD_CFG2, MASK_2REG(DAIF_RFPLL_PD_TXDAC_ME, 1, DAIF_RFPLL_PD_TXDAC_MO, 1));
            break;
        case OM24G_DEVIATION_125K:
            om24g_env.deviation_below_250K = true;
            REGW0(&OM_DAIF->PLL_CTRL1, DAIF_FREQ_DEVIA_BYPASS_MASK); //freq devia bypass=0d
            REGW(&OM_DAIF->PLL_CTRL1, MASK_1REG(DAIF_FREQ_DEVIA_COEFF, 0xFF)); // TX deviation value = 125k
            REGW(&OM_DAIF->KDCO_LUT_2M_REG0, MASK_1REG(DAIF_KDCO_LUT_2M_0, om24g_set_kdco((om24g_env.reg_store.KDCO_LUT_2M_REG0 & DAIF_KDCO_LUT_2M_0_MASK) >> DAIF_KDCO_LUT_2M_0_POS)));
            REGW(&OM_DAIF->KDCO_LUT_2M_REG0, MASK_1REG(DAIF_KDCO_LUT_2M_1, om24g_set_kdco((om24g_env.reg_store.KDCO_LUT_2M_REG0 & DAIF_KDCO_LUT_2M_1_MASK) >> DAIF_KDCO_LUT_2M_1_POS)));
            REGW(&OM_DAIF->KDCO_LUT_2M_REG0, MASK_1REG(DAIF_KDCO_LUT_2M_2, om24g_set_kdco((om24g_env.reg_store.KDCO_LUT_2M_REG0 & DAIF_KDCO_LUT_2M_2_MASK) >> DAIF_KDCO_LUT_2M_2_POS)));
            REGW(&OM_DAIF->KDCO_LUT_2M_REG0, MASK_1REG(DAIF_KDCO_LUT_2M_3, om24g_set_kdco((om24g_env.reg_store.KDCO_LUT_2M_REG0 & DAIF_KDCO_LUT_2M_3_MASK) >> DAIF_KDCO_LUT_2M_3_POS)));
            REGW(&OM_DAIF->KDCO_LUT_2M_REG1, MASK_1REG(DAIF_KDCO_LUT_2M_4, om24g_set_kdco((om24g_env.reg_store.KDCO_LUT_2M_REG1 & DAIF_KDCO_LUT_2M_4_MASK) >> DAIF_KDCO_LUT_2M_4_POS)));
            REGW(&OM_DAIF->KDCO_LUT_2M_REG1, MASK_1REG(DAIF_KDCO_LUT_2M_5, om24g_set_kdco((om24g_env.reg_store.KDCO_LUT_2M_REG1 & DAIF_KDCO_LUT_2M_5_MASK) >> DAIF_KDCO_LUT_2M_5_POS)));
            break;
        case OM24G_DEVIATION_250K:
        case OM24G_DEVIATION_500K:
            om24g_env.deviation_below_250K = false;
            REGW1(&OM_DAIF->PLL_CTRL1, DAIF_FREQ_DEVIA_BYPASS_MASK); //freq devia bypass=1d
            if (SYS_IS_A3()) {   // A3 CHIP
                REGW(&OM_DAIF->PLL_CTRL2, MASK_1REG(DAIF_EN_KVCO2, 0x3));
            } else {  // A2 CHIP
                REGW(&OM_DAIF->PLL_CTRL2, MASK_1REG(DAIF_EN_KVCO2, 0x7));
            } 
            REGW(&OM_DAIF->VCO_CTRL1, MASK_2REG(DAIF_BLE_1M_2M_SEL_ME, 0x0, DAIF_BLE_1M_2M_SEL_MO, 0x0));
            OM_DAIF->KDCO_LUT_2M_REG0 = om24g_env.reg_store.KDCO_LUT_2M_REG0;
            OM_DAIF->KDCO_LUT_2M_REG1 = om24g_env.reg_store.KDCO_LUT_2M_REG1;
            break; 
        default:
            break;
    }
}

// /*******************************************************************************
//  * PUBLIC FUNCTIONS
//  */
void om24g_register_event_callback(drv_isr_callback_t event_cb)
{
    om24g_env.event_cb = event_cb;
}

__RAM_CODES("24G")
void om24g_switch_role(om24g_role_t role)
{
    OM24G_CE_LOW();
    OM24G_FLUSH_TX_FIFO();
    OM24G_FLUSH_RX_FIFO();
    om_error_t ret; // Max: 150us
    DRV_WAIT_US_UNTIL_TO(!((REGR(&OM_24G->STATE, MASK_POS(OM24G_MAIN_STATE)) == 0x02) || (REGR(&OM_24G->STATE, MASK_POS(OM24G_MAIN_STATE)) == 0x00)), 150, ret);(void) ret;
    DRV_DELAY_US(1);
    if (role == OM24G_ROLE_PTX) {
        REGW0(&OM_24G->PKTCTRL0, OM24G_PKTCTRL0_PRIM_RX_MASK);
    } else {
        REGW1(&OM_24G->PKTCTRL0, OM24G_PKTCTRL0_PRIM_RX_MASK);
    }
    DRV_DELAY_US(1);
}

__RAM_CODES("24G")
static void om24g_write_tx_payload(uint16_t dma_tx_num, const uint8_t *payload)
{
    OM_24G->DMA_TX_ADDR = (uint32_t)payload;
    OM_24G->DMA_TX_LEN = dma_tx_num;
}

__RAM_CODES("24G")
bool om24g_write(const uint8_t *tx_payload, uint16_t tx_num)
{
    uint8_t status = 0x00;
    
    if (om24g_env.deviation_below_250K) {   
        REGW(&OM_DAIF->PLL_CTRL2, MASK_1REG(DAIF_EN_KVCO2, 0x0));
        REGW(&OM_DAIF->VCO_CTRL1, MASK_2REG(DAIF_BLE_1M_2M_SEL_ME, 0x1, DAIF_BLE_1M_2M_SEL_MO, 0x1));
    }
    #if RTE_OM24G_SILICONLAB
    REGW(&OM_PHY->TX_CTRL0, MASK_2REG(PHY_TX_CTRL0_EN_INTERP, 0x1, PHY_TX_CTRL0_BDR_PPM_TX, 0x220));
    #endif
    om24g_switch_role(OM24G_ROLE_PTX);
    om24g_write_tx_payload(tx_num, tx_payload);
    om24g_ce_high_pulse();
    while (1) {
        status = OM_24G->INT_ST;
        if ((OM24G_STATUS_TX_DS & status) || (OM24G_STATUS_MAX_RT & status)) {
            OM_24G->INT_ST = status;
            if (OM24G_STATUS_TX_DS & status) {
                status = 0;
                return true;
            } else {
                status = 0;
                return false;
            }
        }
    }
}

__RAM_CODES("24G")
void om24g_write_int(const uint8_t *tx_payload, uint16_t tx_num)
{
    if (om24g_env.deviation_below_250K) {
        REGW(&OM_DAIF->PLL_CTRL2, MASK_1REG(DAIF_EN_KVCO2, 0x0));
        REGW(&OM_DAIF->VCO_CTRL1, MASK_2REG(DAIF_BLE_1M_2M_SEL_ME, 0x1, DAIF_BLE_1M_2M_SEL_MO, 0x1));
    }
    #if RTE_OM24G_SILICONLAB
    REGW(&OM_PHY->TX_CTRL0, MASK_2REG(PHY_TX_CTRL0_EN_INTERP, 0x1, PHY_TX_CTRL0_BDR_PPM_TX, 0x220));
    #endif
    om24g_switch_role(OM24G_ROLE_PTX);
    om24g_write_tx_payload(tx_num, tx_payload);
    om24g_ce_high_pulse();
}

__RAM_CODES("24G")
uint16_t om24g_read(uint8_t *rx_payload, uint32_t timeout_ms)
{
    uint16_t rx_cnt = 0;
    bool rx_right_flag = false;
    om_error_t error;

    if (om24g_env.deviation_below_250K) {
        REGW(&OM_DAIF->VCO_CTRL1, MASK_2REG(DAIF_BLE_1M_2M_SEL_ME, 0x0, DAIF_BLE_1M_2M_SEL_MO, 0x0));   
    }
    OM_24G->DMA_RX_ADDR = (uint32_t)rx_payload;
    om24g_switch_role(OM24G_ROLE_PRX);
    OM24G_CE_HIGH();

    while (1) {
        DRV_WAIT_MS_UNTIL_TO(!( OM_24G->INT_ST & (OM24G_STATUS_RX_DR | OM24G_STATUS_CRC_ERR)), timeout_ms, error);
        if (error != OM_ERROR_OK) {
            OM24G_CE_LOW();
            return 0;
        }
        if (OM24G_STATUS_RX_DR & OM_24G->INT_ST) {
            REGW1(&OM_24G->INT_ST, OM24G_STATUS_RX_DR);
            OM24G_CE_LOW();
            if((!(OM_PHY->FYSNC_DET_INFO & PHY_FYSNC_DET_INFO_FYSNC_CFO_EST_MASK)) && (!(OM24G_STATUS_CRC_ERR & OM_24G->INT_ST))) {
                rx_cnt = om24g_read_rx_payload_width();
                rx_right_flag = true;
            } else {
                rx_right_flag = false;
                OM24G_CE_HIGH();
            }
        }
        if(rx_right_flag) {
            break;
        }
        if (OM24G_STATUS_CRC_ERR & OM_24G->INT_ST) {
            REGW1(&OM_24G->INT_ST, OM24G_STATUS_CRC_ERR);
        }
    }
    return rx_cnt;
}

/**
 *******************************************************************************
 * @brief Receive data from receive FIFO by interrupt mode. The received data and length can be obtained from the callback function.
  
 * @param[in] rx_payload      Data read from receive buffer.
 * @param[in] max_rx_num      The maximum number of receives for this application.

 *******************************************************************************
 */
__RAM_CODES("24G")
void om24g_read_int(uint8_t *rx_payload, uint16_t max_rx_num)
{
    om24g_env.rx_buf = rx_payload;
    om24g_env.max_rx_num = max_rx_num;

    if (om24g_env.deviation_below_250K) { 
        REGW(&OM_DAIF->VCO_CTRL1, MASK_2REG(DAIF_BLE_1M_2M_SEL_ME, 0x0, DAIF_BLE_1M_2M_SEL_MO, 0x0));   
    }

    om24g_switch_role(OM24G_ROLE_PRX);
    OM24G_CE_HIGH();
}

uint16_t om24g_read_ack(void)
{
    uint16_t payload_lenth = 0;

    payload_lenth = om24g_read_rx_payload_width();
    OM24G_CLEAR_ALL_IRQ();

    return payload_lenth;
}

void om24g_set_addr(uint8_t pipenum, uint8_t addr)
{
    switch (pipenum) {
        case OM24G_PIPE_TX:
            REGW(&OM_24G->TX_ADDR, MASK_1REG(OM24G_TX_ADDR, addr));
            break;
        case OM24G_PIPE0:
            REGW(&OM_24G->RX_ADDR_P0, MASK_1REG(OM24G_RX_ADDR_P0, addr));
            break;
        case OM24G_PIPE1:
            REGW(&OM_24G->RX_ADDR_P1, MASK_1REG(OM24G_RX_ADDR_P1, addr));
            break;
        case OM24G_PIPE2:
            REGW(&OM_24G->RX_ADDR_P2, MASK_1REG(OM24G_RX_ADDR_P2, addr));
            break;
        case OM24G_PIPE3:
            REGW(&OM_24G->RX_ADDR_P3, MASK_1REG(OM24G_RX_ADDR_P3, addr));
            break;
        case OM24G_PIPE4:
            REGW(&OM_24G->RX_ADDR_P4, MASK_1REG(OM24G_RX_ADDR_P4, addr));
            break;
        case OM24G_PIPE5:
            REGW(&OM_24G->RX_ADDR_P5, MASK_1REG(OM24G_RX_ADDR_P5, addr));
            break;
        case OM24G_PIPE6:
            REGW(&OM_24G->RX_ADDR_P6, MASK_1REG(OM24G_RX_ADDR_P6, addr));
            break;
        case OM24G_PIPE7:
            REGW(&OM_24G->RX_ADDR_P7, MASK_1REG(OM24G_RX_ADDR_P7, addr));
            break;
        default:
            OM_ASSERT(0);
            break;
    }

}

/*Please use function om24g_set_rf_parameters to set the rate, om24g_set_rate() did not configure the frequency offset.*/
void om24g_set_rate(uint8_t data_rate, bool en_arb, uint8_t n_avr, uint8_t n_rep)
{

    #if RTE_OM24G_SILICONLAB
    REGW(&OM_24G->RF_DR, MASK_4REG(OM24G_EN_RX_ARB, true, OM24G_EN_TX_ARB, false, OM24G_N_AVR, 2, OM24G_N_REP, 2));
    REGW(&OM_24G->RF_DR, MASK_2REG(OM24G_RF_DR_TX_RF_DR, OM24G_RATE_250K, OM24G_RF_DR_RX_RF_DR, OM24G_RATE_500K));
    #else
    REGW(&OM_24G->RF_DR, MASK_4REG(OM24G_EN_RX_ARB, en_arb, OM24G_EN_TX_ARB, en_arb, OM24G_N_AVR, n_avr, OM24G_N_REP, n_rep));
    REGW(&OM_24G->RF_DR, MASK_2REG(OM24G_RF_DR_TX_RF_DR, data_rate, OM24G_RF_DR_RX_RF_DR, data_rate));
    #endif
    if (((data_rate == OM24G_RATE_1M) || (data_rate == OM24G_RATE_2M)) && !en_arb) {
        REGW1(&OM_DAIF->PLL_CTRL1, DAIF_FREQ_DEVIA_BYPASS_MASK); //freq devia bypass=1d
    } else {
        REGW0(&OM_DAIF->PLL_CTRL1, DAIF_FREQ_DEVIA_BYPASS_MASK); //freq devia bypass=0d
    }
}

/*The speed only allows the use of speeds listed in enumeration om24g_rate_t in drv_24g.h. Please do not configure them arbitrarily. 
For special requirements, please contact the developer.*/
void om24g_set_rf_parameters(om24g_rate_t data_rate, uint16_t frequency, float fract_freq)
{
    om24g_set_freq(frequency, fract_freq);
    switch (data_rate) {
        case OM24G_RATE_25K:
            om24g_set_rate(OM24G_RATE_250K, true, 10, 10);
            if (SYS_IS_FPGA()) {
                OM_24G->FPGA_TX_INDEX = 0x02;
            } else {
                om24g_set_deviation(OM24G_DEVIATION_62P5K);
            }
            REGW(&OM_PHY->H_RX_CTRL, MASK_2REG(PHY_H_RX_CTRL_FXP, 0x40, PHY_H_RX_CTRL_RVS_FXP, 0x80)); // RX modulation index = 0.5
            break;
        case OM24G_RATE_50K:
            om24g_set_rate(OM24G_RATE_500K, true, 10, 10);
            if (SYS_IS_FPGA()) {
                OM_24G->FPGA_TX_INDEX = 0x02;
            } else {
                om24g_set_deviation(OM24G_DEVIATION_125K);
            }
            REGW(&OM_PHY->H_RX_CTRL, MASK_2REG(PHY_H_RX_CTRL_FXP, 0x40, PHY_H_RX_CTRL_RVS_FXP, 0x80)); // RX modulation index = 0.5
            break;
        case OM24G_RATE_100K:
            om24g_set_rate(OM24G_RATE_500K, true, 5, 5);
            if (SYS_IS_FPGA()) {
                OM_24G->FPGA_TX_INDEX = 0x03;
            } else {
                om24g_set_deviation(OM24G_DEVIATION_125K);
            }
            REGW(&OM_PHY->H_RX_CTRL, MASK_2REG(PHY_H_RX_CTRL_FXP, 0x60, PHY_H_RX_CTRL_RVS_FXP, 0x55)); //RX modulation index = 0.75
            break;
        case OM24G_RATE_125K:
            om24g_set_rate(OM24G_RATE_500K, true, 4, 4); //  500K/4 = 125k
            if (SYS_IS_FPGA()) {
                OM_24G->FPGA_TX_INDEX = 0x03;
            } else {
                om24g_set_deviation(OM24G_DEVIATION_125K);
            }
            REGW(&OM_PHY->H_RX_CTRL, MASK_2REG(PHY_H_RX_CTRL_FXP, 0x60, PHY_H_RX_CTRL_RVS_FXP, 0x55)); //RX modulation index = 0.75
            break;
        case OM24G_RATE_250K:
            om24g_set_rate(OM24G_RATE_250K, false, 0, 0);
            if (SYS_IS_FPGA()) {
                OM_24G->FPGA_TX_INDEX = 0x03;
            } else {
                om24g_set_deviation(OM24G_DEVIATION_125K);
            }
            REGW(&OM_PHY->H_RX_CTRL, MASK_2REG(PHY_H_RX_CTRL_FXP, 0x60, PHY_H_RX_CTRL_RVS_FXP, 0x55)); //RX modulation index = 0.75

            break;
        case OM24G_RATE_500K:
            om24g_set_rate(data_rate, 0, 0, 0);
            if (SYS_IS_FPGA()) {
                OM_24G->FPGA_TX_INDEX = 0x03;
            } else {
                om24g_set_deviation(OM24G_DEVIATION_125K);
            }
            REGW(&OM_PHY->H_RX_CTRL, MASK_2REG(PHY_H_RX_CTRL_FXP, 0x60, PHY_H_RX_CTRL_RVS_FXP, 0x55)); // RX modulation index = 0.75
            break;
        default:
            om24g_set_rate(data_rate, 0, 0, 0);
            #if RTE_OM24G_NORDIC
                if (SYS_IS_FPGA()) {
                    OM_24G->FPGA_TX_INDEX = 0x01;
                }
                REGW(&OM_PHY->H_RX_CTRL, MASK_2REG(PHY_H_RX_CTRL_FXP, 0x29, PHY_H_RX_CTRL_RVS_FXP, 0xC8)); // RX modulation index = 0.32
            #else
                if (SYS_IS_FPGA()) {
                    OM_24G->FPGA_TX_INDEX = 0x02;
                }
                REGW(&OM_PHY->H_RX_CTRL, MASK_2REG(PHY_H_RX_CTRL_FXP, 0x40, PHY_H_RX_CTRL_RVS_FXP, 0x80)); // RX modulation index = 0.5
                //REGW(&OM_PHY->H_RX_CTRL, MASK_2REG(PHY_H_RX_CTRL_FXP, 0x29, PHY_H_RX_CTRL_RVS_FXP, 0xC8)); // RX modulation index = 0.32
                //REGW(&OM_PHY->H_RX_CTRL, MASK_2REG(PHY_H_RX_CTRL_FXP, 0x60, PHY_H_RX_CTRL_RVS_FXP, 0x55)); //RX modulation index = 0.75
                //REGW(&OM_PHY->H_RX_CTRL, MASK_2REG(PHY_H_RX_CTRL_FXP, 0x80, PHY_H_RX_CTRL_RVS_FXP, 0x40)); //RX modulation index = 1.0
            #endif
            om24g_set_deviation(OM24G_DEVIATION_250K);
            break;
    }
}

void om24g_detection_mode(uint8_t detc_mode)
{
    switch (detc_mode) {
        case OM24G_SOFT_DETECTION:  // 0.5 0.32  support 1M,2M 500K,250K, not support LOW RATE.  highest accruacy, highest latency. Soft demodulation does not support phy mode 1 or above.
            REGW(&OM_PHY->DET_MODE, MASK_1REG(PHY_DET_MODE_DET_MODE, 0x01));
            break;
        case OM24G_HARD_DETECTION: //   suport all modulation index 1M,500K,250K ,LOW RATE,  2M not suport, middle accruacy, middle latency.
            REGW(&OM_PHY->DET_MODE, MASK_1REG(PHY_DET_MODE_DET_MODE, 0x00));
            break;
        case OM24G_DPLL_DETECTION:     // support all modulation index and rate, lowest accruacy, lowest latency.
            REGW(&OM_PHY->DET_MODE, MASK_1REG(PHY_DET_MODE_DET_MODE, 0x03));
            break;
        default:
            OM_ASSERT(0);
            break;
    }
    #if RTE_OM24G_SILICONLAB
    REGW(&OM_PHY->FD_CFO_CMP, MASK_1REG(PHY_FD_CFO_CMP_FD_CFO_CMP, 0));
    REGW(&OM_PHY->TX_CTRL0, MASK_2REG(PHY_TX_CTRL0_TX_GFSK_MODE, 0x04, PHY_TX_CTRL0_COEF_GFSK, 0x0D)); // phy mode4
    REGW(&OM_PHY->TX_CTRL0, MASK_2REG(PHY_TX_CTRL0_EN_INTERP, 0x01, PHY_TX_CTRL0_BDR_PPM_TX, 0x00220)); // 0x00100 0x00520
    REGW(&OM_PHY->STR_CTRL, MASK_1REG(PHY_STR_CTRL_BDR_PPM_RX, 0x0088));
    #else
    REGW(&OM_PHY->FD_CFO_CMP, MASK_1REG(PHY_FD_CFO_CMP_FD_CFO_CMP, 1));
    REGW(&OM_PHY->TX_CTRL0, MASK_1REG(PHY_TX_CTRL0_TX_GFSK_MODE, 0x00)); // phy mode0: GFSK MODE
    #endif
}

int8_t om24g_get_rssi(void)
{
    int8_t rssi;

    rssi = OM_PHY->SIG_DBM_EST_O - 10;

    return rssi;
}

void om24g_set_freq(uint16_t freq, float fract_freq)
{
    uint32_t fract_value;

    REGW(&OM_DAIF->FREQ_CFG0, MASK_1REG(DAIF_FREQ_REG_MO, freq));
    if (fract_freq) {
        fract_value = (uint32_t)(fract_freq * 0x3FFFFF);
        REGW(&OM_DAIF->FREQ_CFG3, MASK_2REG(DAIF_FREQ_FRAC_REG, fract_value, DAIF_FREQ_0P5_EN, 1));
    } else {
        REGW(&OM_DAIF->FREQ_CFG3, MASK_2REG(DAIF_FREQ_FRAC_REG, 0, DAIF_FREQ_0P5_EN, 0));
    }
}

// If the deviation is dynamically switched ,it is prohibited to repeatedly call om24g_init.
void om24g_init(const om24g_config_t *cfg)
{
    // RF clock open
    drv_pmu_ana_enable(true, PMU_ANA_RF_24G);
    om24g_control(OM24G_CONTROL_CLK_ENABLE, NULL);
    #if RTE_OM24G_SILICONLAB
    REGW(&OM_24G->PKTCTRL0, MASK_3REG(OM24G_PKTCTRL0_STRUCT_SEL, cfg->packet_struct_sel, OM24G_PKTCTRL0_RX_0_1_RVS, 1, OM24G_PKTCTRL0_TX_0_1_RVS, 1));
    #else
    REGW(&OM_24G->PKTCTRL0, MASK_3REG(OM24G_PKTCTRL0_STRUCT_SEL, cfg->packet_struct_sel, OM24G_PKTCTRL0_RX_0_1_RVS, 0, OM24G_PKTCTRL0_TX_0_1_RVS, 0));
    #endif
    REGW(&OM_24G->PKTCTRL0, MASK_1REG(OM24G_PKTCTRL0_CE_H_THRE, 1));
    REGW(&OM_24G->PRE_GUARD, MASK_5REG(OM24G_PREGRD_CNT, 0x07, OM24G_PREGRD_EN, 0x00, OM24G_TAIL_CTL, 0x10, OM24G_GUARD_EN, 0x00, OM24G_TAIL_PATERN, 0X02));
    OM_24G->PREAMBLE = cfg->preamble;
    OM_24G->PREAMBLE_LEN = cfg->preamble_len;
    OM_24G->SYNC_WORD_SEL = cfg->sync_word_sel;
    OM_24G->SYNC_WORD0 = cfg->sync_word0;
    OM_24G->SYNC_WORD1 = cfg->sync_word1;
    OM_24G->TX_ADDR = cfg->tx_addr;
    OM_24G->RX_ADDR_P0 = cfg->rx_addr;
    REGW(&OM_24G->PKTCTRL0, MASK_2REG(OM24G_PKTCTRL0_NUM_HDR_BITS, cfg->hdr_bits, OM24G_PKTCTRL0_ADDR_CHK, cfg->addr_chk));
    OM_24G->PACKET_LEN = cfg->static_len;
    REGW(&OM_24G->FB_PKTCTRL, MASK_5REG(OM24G_FB_PKTCTRL_NUM_ADDR1_BITS, cfg->addr1_bits, OM24G_FB_PKTCTRL_NUM_LEN_BITS, cfg->len_bits, OM24G_FB_PKTCTRL_ADDR1_POS, cfg->addr1_pos, OM24G_FB_PKTCTRL_LEN_POS, cfg->len_pos, OM24G_FB_PKTCTRL_ADDR1_LOC, cfg->addr1_loc));
    
    REGW(&OM_24G->EN_AA, 0xFF, (cfg->ack_en ? 0xFF : 0));
    REGW(&OM_24G->FEATURE, MASK_2REG(OM24G_EN_DPL, (cfg->dpl_en ? 1 : 0), OM24G_EN_ACK_PAY, (cfg->ack_en ? 1 : 0)));
    REGW(&OM_24G->DYNPD, 0xFF, (cfg->dpl_en ? 0xFF : 0));

    REGW(&OM_24G->WHITECFG, MASK_4REG(OM24G_WHITE_EN, cfg->white_en, OM24G_WHITE_SKIP_HEADER, cfg->white_skip_hdr, OM24G_WHITE_SKIP_ADDR, cfg->white_skip_addr, OM24G_WHITE_SKIP_CRC, cfg->white_skip_crc));
    OM_24G->WHITESEL = cfg->white_sel;
    OM_24G->WHITESEED = cfg->white_seed;
    OM_24G->WHITEOUT = cfg->white_obit;

    REGW(&OM_24G->CRCCFG, MASK_3REG(OM24G_CRC_EN, cfg->crc_en, OM24G_CRC_LEN, cfg->crc_len, OM24G_CRC_MODE, cfg->crc_mode));
    if (cfg->endian == OM24G_ENDIAN_MSB) {
        #if RTE_OM24G_SILICONLAB
        REGW(&OM_24G->ENDIAN, MASK_7REG(OM24G_EDIBIT_CRC, OM24G_ENDIAN_MSB, OM24G_EDIBIT_PL, OM24G_ENDIAN_MSB, OM24G_EDIBYTE_SW, OM24G_ENDIAN_LSB, OM24G_EDIBIT_SW, OM24G_ENDIAN_MSB, OM24G_EDIBIT_ADDR, OM24G_ENDIAN_MSB, OM24G_EDIBIT_HDR, OM24G_ENDIAN_MSB, OM24G_EDIBYTE_CRC, OM24G_ENDIAN_MSB));
        #else
        REGW(&OM_24G->ENDIAN, MASK_7REG(OM24G_EDIBIT_CRC, OM24G_ENDIAN_MSB, OM24G_EDIBIT_PL, OM24G_ENDIAN_MSB, OM24G_EDIBYTE_SW, OM24G_ENDIAN_MSB, OM24G_EDIBIT_SW, OM24G_ENDIAN_MSB, OM24G_EDIBIT_ADDR, OM24G_ENDIAN_MSB, OM24G_EDIBIT_HDR, OM24G_ENDIAN_MSB, OM24G_EDIBYTE_CRC, OM24G_ENDIAN_MSB));
        #endif
    } else {
        REGW(&OM_24G->ENDIAN, MASK_7REG(OM24G_EDIBIT_CRC, OM24G_ENDIAN_MSB, OM24G_EDIBIT_PL, OM24G_ENDIAN_LSB, OM24G_EDIBYTE_SW, OM24G_ENDIAN_LSB, OM24G_EDIBIT_SW, OM24G_ENDIAN_LSB, OM24G_EDIBIT_ADDR, OM24G_ENDIAN_LSB, OM24G_EDIBIT_HDR, OM24G_ENDIAN_LSB, OM24G_EDIBYTE_CRC, OM24G_ENDIAN_MSB));
    }
    OM_24G->CRCPOLY = cfg->crc_poly;
    OM_24G->CRCINIT = cfg->crc_init;
    REGW(&OM_24G->CRCSKIP, MASK_3REG(OM24G_CRC_SKIP_SYNC, cfg->crc_skip_sync, OM24G_CRC_SKIP_LEN, cfg->crc_skip_len, OM24G_CRC_SKIP_ADDR, cfg->crc_skip_addr));

    REGW(&OM_24G->FA_SETUP_RETR, MASK_2REG(OM24G_FA_SETUP_RETR_ARC, 0x03, OM24G_FA_SETUP_RETR_ARD, 0x00));
    OM_24G->DMA_RX_ADDR = (uint32_t)cfg->rx_data;
    OM_24G->DMA_TX_ADDR = (uint32_t)cfg->tx_data;
    om24g_env.rx_buf = cfg->rx_data;
    REGW(&OM_DAIF->MAIN_ST_CFG1, MASK_1REG(DAIF_TX_WAIT, 0));
#if PLL_CONTIMUE_OPEN
    REGW(&OM_DAIF->MAIN_ST_CFG2, MASK_1REG(DAIF_RX_PLL_WAIT, 0xA0));  // PLL calib = 10us
    #if 1 //Single send or single receive mode
    OM_DAIF->MAIN_ST_CFG0 = 0; 
    REGW(&OM_DAIF->MAIN_ST_CFG1, MASK_1REG(DAIF_RXLDO_WAIT, 0x190));  // LDO calib = 25us
    DRV_DELAY_US(80);
    REGW(&OM_24G->SETUP_VALUE, MASK_3REG(OM24G_RX_SETUP_VALUE, 0x28, OM24G_TX_SETUP_VALUE, 0x1A, OM24G_RX_TM_CNT, 0xFF));
    #else //Receive and transmit conversion in ACK mode
     OM_DAIF->MAIN_ST_CFG0 = 0X00a00140;
    DRV_DELAY_US(80);
    REGW(&OM_24G->SETUP_VALUE, MASK_3REG(OM24G_RX_SETUP_VALUE, 0x3A, OM24G_TX_SETUP_VALUE, 0x33, OM24G_RX_TM_CNT, 0xFF));
    #endif
#else
    //Communicate with NORDIC: 1M RX:6A, TX:63  250k RX:78, TX:63, nordic_ack_timeout:300. 2M: RX:6A, TX:6F
    REGW(&OM_24G->SETUP_VALUE, MASK_3REG(OM24G_RX_SETUP_VALUE, 0x62, OM24G_TX_SETUP_VALUE, 0x5B, OM24G_RX_TM_CNT, 0xFF));
#endif
    //GFSK OR FSK
    if (cfg -> modulation_mode) {
        REGW1(&OM_PHY->TX_CTRL0, PHY_TX_CTRL0_BP_GAU_MASK); //FSK
    } else {
        REGW0(&OM_PHY->TX_CTRL0, PHY_TX_CTRL0_BP_GAU_MASK);
    }
    om24g_detection_mode(cfg->detect_mode);
    //sync word: symble-bit-error criterion
    //REGW0(&OM_PHY->RX_GFSK_SYNC_CTRL, PHY_RX_GFSK_SYNC_CTRL_SBE_MAX_TH_32_MASK);
    //REGW(&OM_PHY->RX_GFSK_SYNC_CTRL, MASK_1REG(PHY_RX_GFSK_SYNC_CTRL_SBE_MAX_TH_32, 2));
    // 1 - enable double address synchronization
    REGW(&OM_PHY->RX_GFSK_SYNC_CTRL, MASK_1REG(PHY_RX_GFSK_SYNC_CTRL_EN_2ND_ADDR, 0));
    //0 - All packages are forced to ACK, ignoring NO_ ACK bit.
    REGW(&OM_24G->FA_PKTCTRL, MASK_1REG(OM24G_FA_PKTCTRL_EN_DYN_ACK, 1));
    #if (RTE_PM)
    pm_sleep_store_restore_callback_register(om24g_pm_sleep_store_restore_handler);
    #endif
    NVIC_ClearPendingIRQ(OM24G_RF_IRQn);
    NVIC_SetPriority(OM24G_RF_IRQn, RTE_OM24G_IRQ_PRIORITY);
    NVIC_EnableIRQ(OM24G_RF_IRQn);
    REGW(&OM_24G->RF_PD_AHEAD, MASK_2REG(OM24G_RF_PD_AHEAD_EN, 1, OM24G_RF_PD_AHEAD, 1));

    om24g_env.reg_store.KDCO_LUT_2M_REG0 = OM_DAIF->KDCO_LUT_2M_REG0;
    
    om24g_env.reg_store.KDCO_LUT_2M_REG1 = OM_DAIF->KDCO_LUT_2M_REG1;
    om24g_set_rf_parameters(cfg->data_rate, cfg->freq, 0);
}

__RAM_CODES("24G")
void drv_om24g_isr(void)
{
    uint8_t status = 0;
    uint16_t rx_cnt = 0;
    static uint8_t rx_count = 0;
    static bool rx_right_flag = false;

    DRV_IRQ_BEGIN();
    status = OM_24G->INT_ST;
    //OM_LOG_DEBUG("status=%02x\r\n", status);
    if ((OM24G_STATUS_TX_DS & status)) {
        REGW1(&OM_24G->INT_ST, OM24G_STATUS_TX_DS);
        if (om24g_env.event_cb) {
            om24g_env.event_cb(OM_24G, DRV_EVENT_COMMON_TRANSMIT_COMPLETED, NULL, NULL);
        }
    }
    if (OM24G_STATUS_MAX_RT & status) {
        REGW1(&OM_24G->INT_ST, OM24G_STATUS_MAX_RT);
        if (om24g_env.event_cb) {
            om24g_env.event_cb(OM_24G, DRV_EVENT_COMMON_RX_OVERFLOW, NULL, NULL);
        }
    }
    if (OM24G_STATUS_RX_DR & status) {
        REGW1(&OM_24G->INT_ST, OM24G_STATUS_RX_DR);
        OM_CRITICAL_BEGIN();
        if((!(OM_PHY->FYSNC_DET_INFO & PHY_FYSNC_DET_INFO_FYSNC_CFO_EST_MASK)) && (!(OM24G_STATUS_CRC_ERR & OM_24G->INT_ST))) {
            rx_count = (rx_count + 1) % 2;
            rx_right_flag = true;
            rx_cnt = om24g_read_rx_payload_width();
            OM_24G->DMA_RX_ADDR = (uint32_t)(om24g_env.rx_buf + om24g_env.max_rx_num * rx_count);
        } else {
            rx_right_flag = false;
        }
        OM_CRITICAL_END();
        if (om24g_env.event_cb && rx_right_flag) {
            om24g_env.event_cb(OM_24G, DRV_EVENT_COMMON_RECEIVE_COMPLETED, (void *)(om24g_env.rx_buf + om24g_env.max_rx_num * (rx_count ? 0 : 1)), (void *)((uint32_t)rx_cnt)); 
        }
    }

    if (OM24G_STATUS_TIME_OUT & status) {
        REGW1(&OM_24G->INT_ST, OM24G_STATUS_TIME_OUT);
        if (om24g_env.event_cb) {
            om24g_env.event_cb(OM_24G, DRV_EVENT_COMMON_GENERAL, NULL, NULL);
        }
    }
    if (OM24G_STATUS_CRC_ERR & OM_24G->INT_ST) {
        REGW1(&OM_24G->INT_ST, OM24G_STATUS_CRC_ERR);
    }
    DRV_IRQ_END();
}

/*Before starting RF work, please turn on the clock in the following order: drv_pmu_ana_enable(true, PMU_ANA_RF_24G); om24g_control(OM24G_CONTROL_CLK_ENABLE, NULL);
 After completing RF work, please turn off the clock in the following order:drv_pmu_ana_enable(false, PMU_ANA_RF_24G);om24g_control(OM24G_CONTROL_CLK_DISABLE, NULL);
*/
void *om24g_control(om24g_control_t control, void *argu)
{
    uint32_t ret;

    ret = (uint32_t)OM_ERROR_OK;

    OM_CRITICAL_BEGIN();
    switch (control) {
        case OM24G_CONTROL_CLK_DISABLE:
            {
                DRV_RCC_CLOCK_ENABLE(RCC_CLK_2P4, 1U);
                DRV_RCC_ANA_CLK_ENABLE_NOIRQ();
                OM24G_CE_LOW();
                OM24G_FLUSH_TX_FIFO();
                OM24G_FLUSH_RX_FIFO();
                om_error_t ret; // Max: 150us
                DRV_WAIT_US_UNTIL_TO(!((REGR(&OM_24G->STATE, MASK_POS(OM24G_MAIN_STATE)) == 0x02) || (REGR(&OM_24G->STATE, MASK_POS(OM24G_MAIN_STATE)) == 0x00)), 150, ret); (void) ret;
                // Wait for the digital state machine to finish running before turning off the clock.
                REGW(&OM_DAIF->DBG_REG, MASK_2REG(DAIF_DBG_EN, 1, DAIF_DBG_IDX, 3));
                DRV_WAIT_US_UNTIL_TO((REGR(&OM_DAIF->DBG_REG, MASK_POS(DAIF_DBG_DATA)) & 0xF), 30, ret); (void) ret;
                REGW(&OM_DAIF->DBG_REG, MASK_1REG(DAIF_DBG_EN, 0));
                #if PLL_CONTIMUE_OPEN
                REGW(&OM_DAIF->PD_CFG1, MASK_2REG(DAIF_RFPLL_PD_ALL_ME, 0, DAIF_RFPLL_PD_ALL_MO, 1));
                #endif
                REGW0(&OM_24G->PKTCTRL0, OM24G_PKTCTRL0_MAC_SEL_MASK); // switch phy between 2.4G and ble.
                REGW(&OM_DAIF->FREQ_CFG0, MASK_1REG(DAIF_FREQ_REG_ME, 0));
                REGW1(&OM_DAIF->PLL_CTRL1, DAIF_FREQ_DEVIA_BYPASS_MASK);
                REGW(&OM_DAIF->PD_CFG2, MASK_1REG(DAIF_RFPLL_PD_TXDAC_ME, 0));
                OM_24G->INT_MASK = 0xFF;
                DRV_RCC_ANA_CLK_RESTORE_NOIRQ();
                DRV_RCC_CLOCK_ENABLE(RCC_CLK_2P4, 0U);
            }
            break;
        case OM24G_CONTROL_CLK_ENABLE:
            {
                DRV_RCC_CLOCK_ENABLE(RCC_CLK_DAIF, 1U);
                DRV_RCC_CLOCK_ENABLE(RCC_CLK_2P4, 1U);
                OM24G_CE_LOW();
                REGW1(&OM_24G->PKTCTRL0, OM24G_PKTCTRL0_MAC_SEL_MASK);
                REGW(&OM_DAIF->FREQ_CFG0, MASK_1REG(DAIF_FREQ_REG_ME, 1));
                if(om24g_env.deviation_below_250K) {
                    REGW0(&OM_DAIF->PLL_CTRL1, DAIF_FREQ_DEVIA_BYPASS_MASK);
                }
                if(OM_24G->RF_DR == 0xA4A4) {  // DATA RATE = OM24G_RATE_25K
                    REGW(&OM_DAIF->PD_CFG2, MASK_2REG(DAIF_RFPLL_PD_TXDAC_ME, 1, DAIF_RFPLL_PD_TXDAC_MO, 1));
                }
                // Block unnecessary interrupts
                OM_24G->INT_MASK = 0x1E;
                #if PLL_CONTIMUE_OPEN
                REGW(&OM_DAIF->PD_CFG1, MASK_2REG(DAIF_RFPLL_PD_ALL_ME, 1, DAIF_RFPLL_PD_ALL_MO, 0));
                #endif
            }
            break;
        case OM24G_CONTROL_RESET:
            {
                DRV_RCC_RESET(RCC_CLK_2P4);
                if((uint32_t)argu) {
                    DRV_RCC_RESET(RCC_CLK_PHY);
                }
                REGW1(&OM_24G->PKTCTRL0, OM24G_PKTCTRL0_MAC_SEL_MASK);
            }
            break;
        case OM24G_CONTROL_SWITCH_ROLE:
            {
                om24g_switch_role((om24g_role_t)((uint32_t)argu));
            }
            break;
        case OM24G_CONTROL_DUMP_RF_REGISTER:
            {
                om24g_dump_rf_register();
            }
            break;
        default:
            break;
    }
    OM_CRITICAL_END();

    return (void *)ret;
}

// true: idle  false: tx busy
bool om24g_tx_idle(void)
{
    bool tx_status = false;

    REGW(&OM_DAIF->DBG_REG, MASK_2REG(DAIF_DBG_EN, 1, DAIF_DBG_IDX, 3));
    tx_status = ((REGR(&OM_DAIF->DBG_REG, MASK_POS(DAIF_DBG_DATA)) & 0x20) || (REGR(&OM_24G->STATE, MASK_POS(OM24G_MAIN_STATE)) == 0x03)) ? false : true;
    REGW(&OM_DAIF->DBG_REG, MASK_1REG(DAIF_DBG_EN, 0));

    return tx_status;
}

bool om24g_rx_idle(void)
{
    bool rx_status = false;

    if(!(OM_PHY->FYSNC_DET_INFO & PHY_FYSNC_DET_INFO_FYSNC_CFO_EST_MASK)) {
        rx_status = true;
    }

    return rx_status;
}

#endif /*RTE_OM24G*/

/*-------------------------------------------End Of File---------------------------------------------*/

/** @} */
