/* ----------------------------------------------------------------------------
 * Copyright (c) 2020-2030 BoLing Limited. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *   1. Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright notice,
 *      this list of conditions and the following disclaimer in the documentation
 *      and/or other materials provided with the distribution.
 *   3. Neither the name of BoLing nor the names of its contributors
 *      may be used to endorse or promote products derived from this software
 *      without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * -------------------------------------------------------------------------- */

/**
 * @file     system.c
 * @brief    CMSIS Device System Source File for BL1826 core1
 * @date     7. Jan. 2022
 * @author   BoLing SW Team
 *
 * @version
 * Version 1.0
 *  - Initial release
 *
 * @{
 */


/*******************************************************************************
 * INCLUDES
 */
#include "om_device.h"
#include "om_driver.h"


/**
 *******************************************************************************
 * @brief Exception / Interrupt Vector table
 *******************************************************************************
 */
extern const VECTOR_TABLE_Type __VECTOR_TABLE[];    /*lint !e526*/


/*******************************************************************************
 * PUBLIC FUNCTIONS
 */
/**
 *******************************************************************************
 * @brief System initialization function, called before init ".data", ".bss" and "other section"
 *******************************************************************************
 */
void SystemInit(void)
{
    #if defined(CONFIG_XIP_FLASH_ALL)
    drv_icache_enable();
    #endif

    #if defined (__VTOR_PRESENT) && (__VTOR_PRESENT == 1U)
    SCB->VTOR = (uint32_t) & (__VECTOR_TABLE[0]);
    #endif

    #if defined (__FPU_USED) && (__FPU_USED == 1U)
    SCB->CPACR |= ((3U << 10U*2U) |           /* enable CP10 Full Access */
                   (3U << 11U*2U)  );         /* enable CP11 Full Access */
    #endif

    #ifdef UNALIGNED_SUPPORT_DISABLE
    SCB->CCR |= SCB_CCR_UNALIGN_TRP_Msk;
    #endif

    #if defined (__ARM_FEATURE_CMSE) && (__ARM_FEATURE_CMSE == 3U)
    TZ_SAU_Setup();
    #endif

    drv_pmu_init();
}

/**
 *******************************************************************************
 * @brief  system init post, called after init ".data", ".bss" and "other section"
 *
 * @note  This may be implemented in the system library to import data such as CP/FT calibration.
 *******************************************************************************
 */
void SystemInitPost(void)
{
	void om_license_init(void);
	om_license_init();	
#if defined(CONFIG_LIB_USE)
    // Use lib SystemInitPost
    SystemInitPostLib();
#elif defined(CONFIG_CPFT_DATA_FETCH)
    // CPFT data
    drv_cpft_init();
    // Config system
    SystemConfig();
    // CPFT data
    drv_cpft_uninit();
#else
    // Config system
    SystemConfig();
#endif
}

#if RTE_RC32M
void drv_pmu_rc32m_x2_startup(void)
{
    OM_SYS->MON = 0x0000;
    drv_pmu_topclk_x2_enable(true);
    extern void drv_pmu_topclk_switch_to_rc32m_x2(void);
    drv_pmu_topclk_switch_to_rc32m_x2();
    REGW1(&OM_PMU->MISC_CTRL_1, PMU_MISC_CTRL_1_CRY32M_EN_MASK);
    // Next XTAL32M startup use fast-startup mode.
    REGW0(&OM_PMU->MISC_CTRL, PMU_MISC_CTRL_FIRST_RUN_REG_MASK);
}
#endif


/**
 *******************************************************************************
 * @brief  system config, called in SystemInitPost
 *******************************************************************************
 */
void SystemConfig(void)
{
    // Init PM
    pm_init();

    // Init Pmu timer
    drv_pmu_timer_init();

    // Initialize random process
    srand(drv_rng_get());
    
#if RTE_RC32M
    
	uint8_t *p_rc32m_ctune = (uint8_t*)0x00403014;//RC32MDƵƫFTУ׼
    REGW(&OM_PMU->CLK_CTRL_1, MASK_1REG(PMU_CLK_CTRL_1_CT_OSC32M, *p_rc32m_ctune));	

    drv_pmu_rc32m_x2_startup();
    
#else
    
     // switch to xtal
    drv_pmu_xtal32m_change_param(54); // Change XTAL32M load capacitance
    drv_pmu_xtal32m_startup();        // TOP clock is 32MHz. If xtal32m is used, this function must be called before other functions that depend on xtal32m.
    
    drv_pmu_xtal32m_x2_startup();     // TOP clock is 64MHz. This function depend on xtal32m
    
#endif
     
    drv_rcc_init();
    
    // Init flash
    drv_sfs_probe(DRV_SFS_IFLASH, 2/*freq*/);
    drv_sfs_config(1/*freq*/, DRV_SF_WIDTH_4LINE, DRV_SF_DELAY_AUTO);
}

/** @} */
